/** @file   wingedalien.cpp
 * @brief   Implementation of WingedAlien - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "wingedalien.h" // class's header file
#include "wingedai.h"
#include "soundsamples.h"
#include "GfxManager.h"
#include "gameanims.h"
#include "weapon.h"
#include "consts.h"
#include "www_map.h"
#include "settings.h"
#include "redrawqueue.h"
#include "animplayer.h"
#include "warglobals.h"
#include "gfxid.h"
#include "AnimId.h"
using namespace eng2d;


namespace WeWantWar {


/** Constructor.
 */
WingedAlien::WingedAlien() :
  Alien( ),
  
  m_speedVec( 0, -2.5 )
{
  ObjectID::Type oid = ObjectID::TYPE_WINGEDALIEN;
  
  m_speedVec.y( - Settings::floatObjProp(oid, "spd_walk:") );

	this->setCorrectAnimation( GameAnims::EIdle );
  
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  
  // 4 collisionpoints
  this->setCollisionPoint( 0, Vec2D( 0,-7) );
  this->setCollisionPoint( 1, Vec2D( 0, 7) );
  this->setCollisionPoint( 2, Vec2D( -7, 0) );
  this->setCollisionPoint( 3, Vec2D( 7, 0) );
  
  this->altitude(60);
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  
  // Create the AI-object.
  this->setController( new WingedAI( this ) );
}


/** Destructor.
 */
WingedAlien::~WingedAlien()
{
}



/** The update-method
 */
void WingedAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    float vspd = this->verticalSpeed();
    vspd -= Consts::GRAVITY;
    this->verticalSpeed( vspd );
    
    float alt = this->altitude();
    if ( alt < 1 ) {
      this->setCorrectAnimation( GameAnims::EKilled );
      this->state( GameObject::STATE_KILLED );
      return;
    }
    
    // As the flying alien is dying, we still keep it moving as long as it
    // touches the ground
    Vec2D v = m_speedVec;
    v.rotate( this->angle() );
    if ( m_pController->forward() == 1 ) {
      this->move( v );
    }
    return;
  }
  
  // This alien is still alive.
  m_pController->update();
  
  Vec2D spdV = m_speedVec;
  spdV.rotate( this->angle() );
  if ( m_pController->forward() == 1 ) {
    this->move( spdV );
  }
  this->changeAngle( m_pController->turn() );
  
  if ( m_pController->shoot() == 1 ) {
    this->attack();
  }
}



/** The redraw-method
 */
void WingedAlien::redraw( RedrawQueue* pQueue )
{
  Vec2D p( this->position() );
  BITMAP* pS = m_animation.currentFrame().asBitmap();
  
  int alX = static_cast<int>( p.x() ) - Map::scrollX;
  int alY = static_cast<int>( p.y() ) - Map::scrollY;
  
  alX -= pS->w / 2;
  alY -= pS->h / 2;
  if ( this->state() == GameObject::STATE_KILLED ) {
    pQueue->addRotatedSprite( RedrawQueue::PRI_KILLED, alX,alY,
                              itofix( m_angle ), pS );
    return;
  }

  // This is not dead. We draw the alien at FLYING - layer and shadow
  pQueue->addRotatedSprite( RedrawQueue::PRI_FLYING, alX,alY,
                            itofix( m_angle ), pS );

  
  int rotateIndex = this->angle() / 4;
  BITMAP* shadowSprite = GfxManager::findBitmap( GfxId::KAlienWingedShadow, rotateIndex );
  p = this->position();
  alX = p.intX() - Map::scrollX;
  alY = p.intY() - Map::scrollY;
  int alt = static_cast<int>( this->altitude() / 5.0 );
  alX -= alt;
  alX -= shadowSprite->w / 2;
  alY += alt;
  alY -= shadowSprite->h / 2;
  pQueue->add( RedrawQueue::PRI_ABOVE_NORMAL, alX, alY,
               RedrawQueue::SPTYPE_SPRITE, RedrawQueue::BMODE_MULTIPLY,
               128, shadowSprite );
}



/** Kills this alien
 */
void WingedAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Makes sound
 */
void WingedAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    Sound::playSample(SMP_ALIEN3_SHOOT, false);
  }
  if ( id == GameObject::SND_DIE ) {
    Sound::playSample(SMP_ALIEN3_DIE, false);
  }
}



/** Hit by bullet
 */
bool WingedAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                             9, Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }
  return ret;
}



/** Returns the type id of this object.
 */
ObjectID::Type WingedAlien::objectType() const
{
  return ObjectID::TYPE_WINGEDALIEN;
}



/** Sets the requested animation
 */
void WingedAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation(AnimId::KWingedAlien, aAnimId);
    this->setAnimation( anim, aAnimId );
  }
}



/** Shoots
 */
void WingedAlien::attack()
{
  if ( Settings::difficultyLevel != Settings::KHard ) {
    if ( this->reloading() == true ) {
      return;
    }
  }
  Weapon::Specs& w = Weapon::getWeaponSpecs( Weapon::W_ALIENFIREBALL );

  Bullet* b = BulletTable::createBullet( this, m_position,
                                         Bullet::EAlienFireball );
  WarGlobals::pBulletManager->spawnBullet( b );
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( Alien::RELOAD_COUNTER_INDEX, 1 );
}


} // end of namespace

